<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Acceso Administrativo - Bar VIP</title>
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-color: #2563eb;
            --primary-dark: #1d4ed8;
            --success-color: #10b981;
            --danger-color: #ef4444;
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-400: #9ca3af;
            --gray-500: #6b7280;
            --gray-600: #4b5563;
            --gray-700: #374151;
            --gray-800: #1f2937;
            --gray-900: #111827;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: var(--gray-700);
            background: linear-gradient(135deg, #667eea 0%, #764ba2 50%, #667eea 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            overflow: hidden;
        }

        /* Fondo animado */
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: 
                radial-gradient(circle at 20% 80%, rgba(120, 119, 198, 0.3) 0%, transparent 50%),
                radial-gradient(circle at 80% 20%, rgba(255, 119, 198, 0.3) 0%, transparent 50%),
                radial-gradient(circle at 40% 40%, rgba(120, 219, 255, 0.3) 0%, transparent 50%);
            animation: float 6s ease-in-out infinite;
            pointer-events: none;
        }

        @keyframes float {
            0%, 100% { transform: translateY(0px) rotate(0deg); }
            33% { transform: translateY(-10px) rotate(1deg); }
            66% { transform: translateY(10px) rotate(-1deg); }
        }

        .login-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            padding: 3rem;
            width: 100%;
            max-width: 420px;
            box-shadow: 
                0 20px 40px rgba(0, 0, 0, 0.15),
                0 10px 20px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
            z-index: 10;
            transform: translateY(20px);
            opacity: 0;
            animation: slideIn 0.8s ease forwards;
        }

        @keyframes slideIn {
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }

        .login-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
            animation: shine 3s ease-in-out infinite;
        }

        @keyframes shine {
            0% { left: -100%; }
            100% { left: 100%; }
        }

        .login-header {
            text-align: center;
            margin-bottom: 2rem;
        }

        .login-logo {
            width: 80px;
            height: 80px;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            color: white;
            font-size: 2rem;
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.4);
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }

        .login-title {
            font-size: 1.75rem;
            font-weight: 700;
            color: var(--gray-800);
            margin-bottom: 0.5rem;
        }

        .login-subtitle {
            color: var(--gray-600);
            font-size: 0.95rem;
        }

        .login-form {
            space-y: 1.5rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-label {
            display: block;
            font-weight: 600;
            color: var(--gray-700);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }

        .form-input-group {
            position: relative;
        }

        .form-input {
            width: 100%;
            padding: 1rem 1rem 1rem 3rem;
            border: 2px solid var(--gray-200);
            border-radius: 12px;
            font-size: 1rem;
            background: white;
            transition: all 0.3s ease;
            outline: none;
        }

        .form-input:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 4px rgba(37, 99, 235, 0.1);
            transform: translateY(-1px);
        }

        .form-input.error {
            border-color: var(--danger-color);
            background: #fef2f2;
        }

        .input-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray-400);
            font-size: 1.1rem;
            pointer-events: none;
            transition: color 0.3s ease;
        }

        .form-input:focus + .input-icon {
            color: var(--primary-color);
        }

        .password-toggle {
            position: absolute;
            right: 1rem;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--gray-400);
            cursor: pointer;
            font-size: 1.1rem;
            transition: color 0.3s ease;
        }

        .password-toggle:hover {
            color: var(--gray-600);
        }

        .form-options {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            font-size: 0.9rem;
        }

        .remember-me {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--gray-600);
            cursor: pointer;
        }

        .remember-me input[type="checkbox"] {
            width: 16px;
            height: 16px;
            accent-color: var(--primary-color);
        }

        .forgot-password {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
            transition: color 0.3s ease;
        }

        .forgot-password:hover {
            color: var(--primary-dark);
        }

        .login-button {
            width: 100%;
            padding: 1rem;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            margin-bottom: 1rem;
        }

        .login-button:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.4);
        }

        .login-button:active:not(:disabled) {
            transform: translateY(0);
        }

        .login-button:disabled {
            background: var(--gray-400);
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
        }

        .button-spinner {
            display: none;
            width: 20px;
            height: 20px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-top: 2px solid white;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .login-button.loading .button-spinner {
            display: inline-block;
        }

        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
            font-size: 0.9rem;
            display: none;
            animation: slideDown 0.3s ease;
        }

        @keyframes slideDown {
            from { transform: translateY(-10px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }

        .alert-error {
            background: #fef2f2;
            border: 1px solid #fecaca;
            color: #991b1b;
        }

        .alert-success {
            background: #f0fdf4;
            border: 1px solid #bbf7d0;
            color: #166534;
        }

        .alert-info {
            background: #eff6ff;
            border: 1px solid #bfdbfe;
            color: #1e40af;
        }

        .login-footer {
            text-align: center;
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid var(--gray-200);
            color: var(--gray-500);
            font-size: 0.8rem;
        }

        .demo-credentials {
            background: var(--gray-50);
            border: 1px solid var(--gray-200);
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            font-size: 0.85rem;
        }

        .demo-credentials h4 {
            color: var(--gray-700);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }

        .demo-credentials .credentials {
            display: flex;
            justify-content: space-between;
            font-family: 'Courier New', monospace;
            background: white;
            padding: 0.5rem;
            border-radius: 4px;
            margin: 0.25rem 0;
        }

        /* Responsive */
        @media (max-width: 480px) {
            .login-container {
                margin: 1rem;
                padding: 2rem;
                border-radius: 16px;
            }
            
            .login-title {
                font-size: 1.5rem;
            }
            
            .form-input {
                padding: 0.875rem 0.875rem 0.875rem 2.5rem;
            }
        }

        /* Efectos adicionales */
        .floating-elements {
            position: absolute;
            width: 100%;
            height: 100%;
            overflow: hidden;
            pointer-events: none;
        }

        .floating-element {
            position: absolute;
            opacity: 0.1;
            animation: floatUp 10s infinite ease-in-out;
        }

        @keyframes floatUp {
            0% {
                transform: translateY(100vh) rotate(0deg);
                opacity: 0;
            }
            10% {
                opacity: 0.1;
            }
            90% {
                opacity: 0.1;
            }
            100% {
                transform: translateY(-100px) rotate(360deg);
                opacity: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Elementos flotantes decorativos -->
    <div class="floating-elements">
        <div class="floating-element" style="left: 10%; animation-delay: 0s; font-size: 2rem;">🍸</div>
        <div class="floating-element" style="left: 20%; animation-delay: 2s; font-size: 1.5rem;">🎵</div>
        <div class="floating-element" style="left: 30%; animation-delay: 4s; font-size: 2.5rem;">🎉</div>
        <div class="floating-element" style="left: 40%; animation-delay: 1s; font-size: 1.8rem;">✨</div>
        <div class="floating-element" style="left: 50%; animation-delay: 3s; font-size: 2rem;">🍾</div>
        <div class="floating-element" style="left: 60%; animation-delay: 5s; font-size: 1.5rem;">🎊</div>
        <div class="floating-element" style="left: 70%; animation-delay: 2.5s; font-size: 2.2rem;">🥂</div>
        <div class="floating-element" style="left: 80%; animation-delay: 4.5s; font-size: 1.7rem;">🌟</div>
        <div class="floating-element" style="left: 90%; animation-delay: 1.5s; font-size: 2rem;">🎭</div>
    </div>

    <div class="login-container">
        <div class="login-header">
            <div class="login-logo">
                <i class="fas fa-cocktail"></i>
            </div>
            <h1 class="login-title">Bar VIP Admin</h1>
            <p class="login-subtitle">Panel de Administración</p>
        </div>

        <!-- Credenciales demo -->
        <div class="demo-credentials">
            <h4><i class="fas fa-info-circle"></i> Credenciales Demo</h4>
            <div class="credentials">
                <span>Usuario:</span>
                <strong>admin</strong>
            </div>
            <div class="credentials">
                <span>Contraseña:</span>
                <strong>admin123</strong>
            </div>
        </div>

        <!-- Alertas -->
        <div id="alertContainer">
            <div id="alertMessage" class="alert"></div>
        </div>

        <form id="loginForm" class="login-form">
            <div class="form-group">
                <label for="username" class="form-label">Usuario</label>
                <div class="form-input-group">
                    <input type="text" 
                           id="username" 
                           name="username" 
                           class="form-input" 
                           placeholder="Ingresa tu usuario"
                           required
                           autocomplete="username">
                    <i class="fas fa-user input-icon"></i>
                </div>
            </div>

            <div class="form-group">
                <label for="password" class="form-label">Contraseña</label>
                <div class="form-input-group">
                    <input type="password" 
                           id="password" 
                           name="password" 
                           class="form-input" 
                           placeholder="Ingresa tu contraseña"
                           required
                           autocomplete="current-password">
                    <i class="fas fa-lock input-icon"></i>
                    <button type="button" class="password-toggle" id="togglePassword">
                        <i class="fas fa-eye"></i>
                    </button>
                </div>
            </div>

            <div class="form-options">
                <label class="remember-me">
                    <input type="checkbox" id="rememberMe" name="remember">
                    <span>Recordarme</span>
                </label>
                <a href="#" class="forgot-password" id="forgotPassword">¿Olvidaste tu contraseña?</a>
            </div>

            <button type="submit" id="loginButton" class="login-button">
                <div class="button-spinner"></div>
                <span class="button-text">Iniciar Sesión</span>
            </button>
        </form>

        <div class="login-footer">
            <p>&copy; 2025 Bar VIP. Sistema de reconocimiento facial.</p>
            <p>Versión 1.0.0 - Seguro y confiable</p>
        </div>
    </div>

    <script>
        class LoginManager {
            constructor() {
                this.form = document.getElementById('loginForm');
                this.usernameInput = document.getElementById('username');
                this.passwordInput = document.getElementById('password');
                this.loginButton = document.getElementById('loginButton');
                this.alertContainer = document.getElementById('alertMessage');
                this.togglePasswordBtn = document.getElementById('togglePassword');
                
                this.init();
            }

            init() {
                this.setupEventListeners();
                this.checkRememberedUser();
                this.focusFirstInput();
            }

            setupEventListeners() {
                // Form submission
                this.form.addEventListener('submit', (e) => {
                    e.preventDefault();
                    this.handleLogin();
                });

                // Toggle password visibility
                this.togglePasswordBtn.addEventListener('click', () => {
                    this.togglePasswordVisibility();
                });

                // Input validation on blur
                this.usernameInput.addEventListener('blur', () => {
                    this.validateField(this.usernameInput);
                });

                this.passwordInput.addEventListener('blur', () => {
                    this.validateField(this.passwordInput);
                });

                // Clear errors on focus
                this.usernameInput.addEventListener('focus', () => {
                    this.clearFieldError(this.usernameInput);
                });

                this.passwordInput.addEventListener('focus', () => {
                    this.clearFieldError(this.passwordInput);
                });

                // Enter key handling
                document.addEventListener('keydown', (e) => {
                    if (e.key === 'Enter' && !this.loginButton.disabled) {
                        this.form.dispatchEvent(new Event('submit'));
                    }
                });

                // Forgot password
                document.getElementById('forgotPassword').addEventListener('click', (e) => {
                    e.preventDefault();
                    this.handleForgotPassword();
                });

                // Demo credentials click
                document.querySelectorAll('.credentials').forEach(cred => {
                    cred.addEventListener('click', () => {
                        this.fillDemoCredentials();
                    });
                });
            }

            async handleLogin() {
                const username = this.usernameInput.value.trim();
                const password = this.passwordInput.value;
                const remember = document.getElementById('rememberMe').checked;

                // Basic validation
                if (!this.validateForm()) {
                    return;
                }

                this.setLoading(true);
                this.hideAlert();

                try {
                    const response = await fetch('api/auth/login.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            username: username,
                            password: password,
                            remember: remember
                        })
                    });

                    const data = await response.json();

                    if (data.success) {
                        this.showAlert('¡Bienvenido! Redirigiendo...', 'success');
                        
                        // Save user preference
                        if (remember) {
                            localStorage.setItem('remembered_user', username);
                        } else {
                            localStorage.removeItem('remembered_user');
                        }

                        // Redirect after short delay
                        setTimeout(() => {
                            window.location.href = 'index.php';
                        }, 1500);

                    } else {
                        throw new Error(data.message || 'Error de autenticación');
                    }

                } catch (error) {
                    console.error('Login error:', error);
                    this.showAlert(error.message || 'Error de conexión. Inténtalo de nuevo.', 'error');
                    this.shakeContainer();
                } finally {
                    this.setLoading(false);
                }
            }

            validateForm() {
                let isValid = true;

                if (!this.validateField(this.usernameInput)) {
                    isValid = false;
                }

                if (!this.validateField(this.passwordInput)) {
                    isValid = false;
                }

                return isValid;
            }

            validateField(field) {
                const value = field.value.trim();
                let isValid = true;
                let message = '';

                switch (field.id) {
                    case 'username':
                        if (!value) {
                            message = 'El usuario es requerido';
                            isValid = false;
                        } else if (value.length < 2) {
                            message = 'El usuario debe tener al menos 2 caracteres';
                            isValid = false;
                        }
                        break;
                        
                    case 'password':
                        if (!value) {
                            message = 'La contraseña es requerida';
                            isValid = false;
                        } else if (value.length < 3) {
                            message = 'La contraseña debe tener al menos 3 caracteres';
                            isValid = false;
                        }
                        break;
                }

                if (!isValid) {
                    this.showFieldError(field, message);
                } else {
                    this.clearFieldError(field);
                }

                return isValid;
            }

            showFieldError(field, message) {
                field.classList.add('error');
                // You could show the error message in a tooltip or below the field
            }

            clearFieldError(field) {
                field.classList.remove('error');
            }

            togglePasswordVisibility() {
                const type = this.passwordInput.getAttribute('type');
                const icon = this.togglePasswordBtn.querySelector('i');

                if (type === 'password') {
                    this.passwordInput.setAttribute('type', 'text');
                    icon.classList.replace('fa-eye', 'fa-eye-slash');
                } else {
                    this.passwordInput.setAttribute('type', 'password');
                    icon.classList.replace('fa-eye-slash', 'fa-eye');
                }
            }

            setLoading(loading) {
                this.loginButton.disabled = loading;
                this.loginButton.classList.toggle('loading', loading);
                
                const buttonText = this.loginButton.querySelector('.button-text');
                buttonText.textContent = loading ? 'Iniciando sesión...' : 'Iniciar Sesión';
            }

            showAlert(message, type) {
                this.alertContainer.textContent = message;
                this.alertContainer.className = `alert alert-${type}`;
                this.alertContainer.style.display = 'block';
                
                // Auto hide after 5 seconds for success messages
                if (type === 'success') {
                    setTimeout(() => {
                        this.hideAlert();
                    }, 5000);
                }
            }

            hideAlert() {
                this.alertContainer.style.display = 'none';
            }

            shakeContainer() {
                const container = document.querySelector('.login-container');
                container.style.animation = 'none';
                container.offsetHeight; // Trigger reflow
                container.style.animation = 'shake 0.5s ease-in-out';
                
                setTimeout(() => {
                    container.style.animation = '';
                }, 500);
            }

            checkRememberedUser() {
                const rememberedUser = localStorage.getItem('remembered_user');
                if (rememberedUser) {
                    this.usernameInput.value = rememberedUser;
                    document.getElementById('rememberMe').checked = true;
                    this.passwordInput.focus();
                }
            }

            focusFirstInput() {
                if (this.usernameInput.value) {
                    this.passwordInput.focus();
                } else {
                    this.usernameInput.focus();
                }
            }

            fillDemoCredentials() {
                this.usernameInput.value = 'admin';
                this.passwordInput.value = 'admin123';
                this.passwordInput.focus();
                
                this.showAlert('Credenciales demo cargadas. Haz click en "Iniciar Sesión"', 'info');
                
                // Clear any previous errors
                this.clearFieldError(this.usernameInput);
                this.clearFieldError(this.passwordInput);
            }

            handleForgotPassword() {
                this.showAlert(
                    'Para recuperar tu contraseña, contacta al administrador del sistema.',
                    'info'
                );
            }
        }

        // Add shake animation CSS
        const shakeCSS = `
            @keyframes shake {
                0%, 100% { transform: translateX(0) translateY(0); }
                10%, 30%, 50%, 70%, 90% { transform: translateX(-5px) translateY(0); }
                20%, 40%, 60%, 80% { transform: translateX(5px) translateY(0); }
            }
        `;
        
        const style = document.createElement('style');
        style.textContent = shakeCSS;
        document.head.appendChild(style);

        // Initialize login manager when DOM is loaded
        document.addEventListener('DOMContentLoaded', () => {
            new LoginManager();
        });

        // Add some easter eggs
        let clickCount = 0;
        document.querySelector('.login-logo').addEventListener('click', () => {
            clickCount++;
            if (clickCount === 5) {
                document.body.style.animation = 'float 2s ease-in-out infinite';
                setTimeout(() => {
                    document.body.style.animation = '';
                    clickCount = 0;
                }, 10000);
            }
        });
    </script>
</body>
</html>

---

<?php
// /admin/api/auth/login.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

require_once __DIR__ . '/../../../bd/bd.php';

// Rate limiting simple
$max_attempts = 5;
$lockout_time = 300; // 5 minutos

// Verificar intentos por IP
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$cache_key = 'login_attempts_' . md5($ip);

// Función simple de cache en archivos
function getCache($key) {
    $file = sys_get_temp_dir() . '/' . $key;
    if (file_exists($file) && (time() - filemtime($file)) < 300) {
        return json_decode(file_get_contents($file), true);
    }
    return null;
}

function setCache($key, $data) {
    $file = sys_get_temp_dir() . '/' . $key;
    file_put_contents($file, json_encode($data));
}

try {
    // Obtener datos JSON
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Datos no válidos');
    }
    
    $username = trim($input['username'] ?? '');
    $password = $input['password'] ?? '';
    $remember = $input['remember'] ?? false;
    
    if (empty($username) || empty($password)) {
        throw new Exception('Usuario y contraseña son requeridos');
    }
    
    // Verificar rate limiting
    $attempts = getCache($cache_key) ?? ['count' => 0, 'last_attempt' => 0];
    
    if ($attempts['count'] >= $max_attempts) {
        $time_remaining = $lockout_time - (time() - $attempts['last_attempt']);
        if ($time_remaining > 0) {
            throw new Exception("Demasiados intentos fallidos. Intenta de nuevo en " . ceil($time_remaining/60) . " minutos.");
        } else {
            // Reset attempts after lockout period
            $attempts = ['count' => 0, 'last_attempt' => 0];
        }
    }
    
    // Buscar administrador
    $stmt = $pdo->prepare("
        SELECT id, username, password_hash, nombre, email, rol, activo
        FROM admins 
        WHERE username = ?
    ");
    $stmt->execute([$username]);
    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Verificar credenciales
    if (!$admin || !password_verify($password, $admin['password_hash'])) {
        // Incrementar intentos fallidos
        $attempts['count']++;
        $attempts['last_attempt'] = time();
        setCache($cache_key, $attempts);
        
        // Log intento fallido
        $stmt = $pdo->prepare("
            INSERT INTO admin_logs (admin_id, accion, detalles, ip_address, user_agent)
            VALUES (NULL, 'login_fallido', ?, ?, ?)
        ");
        $stmt->execute([
            json_encode(['username' => $username]),
            $ip,
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
        throw new Exception('Usuario o contraseña incorrectos');
    }
    
    // Verificar si está activo
    if (!$admin['activo']) {
        throw new Exception('Usuario desactivado. Contacta al administrador.');
    }
    
    // Reset attempts on successful login
    setCache($cache_key, ['count' => 0, 'last_attempt' => 0]);
    
    // Crear sesión
    session_regenerate_id(true);
    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_id'] = $admin['id'];
    $_SESSION['admin_username'] = $admin['username'];
    $_SESSION['admin_nombre'] = $admin['nombre'];
    $_SESSION['admin_rol'] = $admin['rol'];
    $_SESSION['login_time'] = time();
    
    // Configurar cookie de sesión
    if ($remember) {
        ini_set('session.cookie_lifetime', 30 * 24 * 60 * 60); // 30 días
    } else {
        ini_set('session.cookie_lifetime', 0); // Hasta cerrar navegador
    }
    
    // Actualizar último login
    $stmt = $pdo->prepare("
        UPDATE admins 
        SET ultimo_login = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$admin['id']]);
    
    // Log login exitoso
    $stmt = $pdo->prepare("
        INSERT INTO admin_logs (admin_id, accion, detalles, ip_address, user_agent)
        VALUES (?, 'login_admin', ?, ?, ?)
    ");
    $stmt->execute([
        $admin['id'],
        json_encode([
            'nombre' => $admin['nombre'],
            'rol' => $admin['rol'],
            'remember' => $remember
        ]),
        $ip,
        $_SERVER['HTTP_USER_AGENT'] ?? ''
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Login exitoso',
        'user' => [
            'id' => $admin['id'],
            'username' => $admin['username'],
            'nombre' => $admin['nombre'],
            'rol' => $admin['rol']
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/auth/check.php
session_start();
header('Content-Type: application/json; charset=utf-8');

try {
    if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
        throw new Exception('No autenticado');
    }
    
    $login_time = $_SESSION['login_time'] ?? 0;
    $current_time = time();
    $session_duration = $current_time - $login_time;
    $max_session_time = 8 * 60 * 60; // 8 horas
    $warning_time = $max_session_time - (30 * 60); // Advertir 30 min antes
    
    $remaining_time = $max_session_time - $session_duration;
    
    // Verificar si la sesión ha expirado
    if ($session_duration > $max_session_time) {
        session_destroy();
        throw new Exception('Sesión expirada');
    }
    
    $response = [
        'success' => true,
        'user' => [
            'id' => $_SESSION['admin_id'],
            'username' => $_SESSION['admin_username'],
            'nombre' => $_SESSION['admin_nombre'],
            'rol' => $_SESSION['admin_rol']
        ],
        'session' => [
            'duration' => $session_duration,
            'remaining' => $remaining_time
        ]
    ];
    
    // Advertir si la sesión está por expirar
    if ($remaining_time < 1800) { // 30 minutos
        $response['warning'] = true;
        $response['remaining'] = ceil($remaining_time / 60); // En minutos
    }
    
    echo json_encode($response);
    
} catch (Exception $e) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/auth/extend.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Methods: POST, OPTIONS');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

try {
    if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
        throw new Exception('No autenticado');
    }
    
    // Extender sesión actualizando el tiempo de login
    $_SESSION['login_time'] = time();
    
    require_once __DIR__ . '/../../../bd/bd.php';
    
    // Log extensión de sesión
    $stmt = $pdo->prepare("
        INSERT INTO admin_logs (admin_id, accion, detalles, ip_address)
        VALUES (?, 'session_extended', ?, ?)
    ");
    $stmt->execute([
        $_SESSION['admin_id'],
        json_encode(['extended_at' => date('Y-m-d H:i:s')]),
        $_SERVER['REMOTE_ADDR'] ?? ''
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Sesión extendida exitosamente'
    ]);
    
} catch (Exception $e) {
    http_response_code(401);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/logout.php
session_start();

// Log logout si hay sesión activa
if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
    require_once __DIR__ . '/../bd/bd.php';
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO admin_logs (admin_id, accion, detalles, ip_address)
            VALUES (?, 'logout_admin', ?, ?)
        ");
        $stmt->execute([
            $_SESSION['admin_id'] ?? null,
            json_encode([
                'logout_time' => date('Y-m-d H:i:s'),
                'session_duration' => time() - ($_SESSION['login_time'] ?? time())
            ]),
            $_SERVER['REMOTE_ADDR'] ?? ''
        ]);
    } catch (Exception $e) {
        // Error logging no debe impedir el logout
        error_log('Error logging logout: ' . $e->getMessage());
    }
}

// Destruir sesión
session_destroy();

// Limpiar cookie de sesión
if (isset($_COOKIE[session_name()])) {
    setcookie(session_name(), '', time() - 3600, '/');
}

// Redirigir al login
header('Location: login.php');
exit;
?>

---

<?php
// /admin/auth_check.php - Include this at the top of protected pages
session_start();

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

// Verificar si la sesión ha expirado
$login_time = $_SESSION['login_time'] ?? 0;
$current_time = time();
$session_duration = $current_time - $login_time;
$max_session_time = 8 * 60 * 60; // 8 horas

if ($session_duration > $max_session_time) {
    session_destroy();
    header('Location: login.php?expired=1');
    exit;
}

// Actualizar última actividad (opcional)
$_SESSION['last_activity'] = $current_time;
?>